              
/*****************************************************************************
*                           Freescale Semiconductor                          *
*                   Brazilian Semiconductor Tecnology Center                 *
*                                                                            *
*  Project    : MC56F8006 Modular Pixel Matrix                               *
*  Version    : 0.1                                                          *
*  Date       : 16/Apr/2009                                                  *
*  Engineers  : Alexandre Dias, Bruno Bastos                                 *
*               Humberto Carvalho, Leonardo Mangiapelo                       *
*               Renato Frias                                                 *
*                                                                            *
* -------------------------------------------------------------------------- *
*  Description:                                                              *
*                                                                            *
*    MC56F8006 IIC												 			 * 
*                                                                            *
******************************************************************************/

/************************* Project Include Files *****************************/

#ifndef __lib_iic_c
#define __lib_iic_c

#include "lib_iic.h"
#include "lib_sci.h"

/********************************* Defines ***********************************/

#define EnableIntrp()	 asm(moveu.w #$0000,sr)
/* global variables for I2C driver */
 /* I2C divider table for MCF51xx.
  */
  
/****************************** Prototypes ***********************************/

const uint8_t IIC_prescaler_val[] = 
{    
 20,22,24,26,28,30,34,40,
 28,32,36,40,44,48,56,68,
 48,56,64,72,80,88,104,128,
 80,96,112,128,144,160,192,240,
 160,192,224,256,288,320,384,480,
 320,384,448,512,576,640,768,960,
 640,768,896,1024,1152,1280,1536,1920,
 1280,1536,1792,2048,2304,2560,3072,3840 
};

vint8_t RW_Flag = IIC_RW_WRITE;		/* read or write flag: 0 for write, 1 for read */
vint8_t FACK_Flag = 0;		/* FACK flag */

vint8_t *txp = NULL;	/* pointer to tx buf */
vint8_t *rxp = NULL;	/* pointer to rx buf */
vint8_t *base_txBuf = NULL;
vint8_t *base_rxBuf = NULL;
vint16_t LastRdOffst; 
vint16_t LastWrOffst;
int8_t TData[I2C_TX_BUF_SIZE] = "0123456789";
int8_t RData[I2C_RX_BUF_SIZE] = "";


/*********************** Constants and Variables *****************************/

union
{
    unsigned int Word;

    struct
    {
        unsigned char tx2;
       unsigned char tx1;
    } TX_struct;
} TX_union;

#define TX1 TX_union.TX_struct.tx1
#define TX2 TX_union.TX_struct.tx2
#define TXW TX_union.Word

unsigned char RX = 0;
unsigned char *RX2;

unsigned char Data_Received = 0;
unsigned char Pending_TX1 = 0;
unsigned char Pending_TX2 = 0;

unsigned long Sum = 0;
unsigned int Max  = 0;
unsigned int Min  = 4095;

unsigned char State = 20;

unsigned int Samples = 0;
unsigned int Remaining = 0;

volatile char rxbuffer[7];
volatile char rx_buf_read_pointer=0;
volatile char tempData=0;
 





/*****************************************************************************
* Funcion Name  : InitPorts
* Parameters    : none
* Returns       : none
* Notes         : 
******************************************************************************/

void srw (void)
{
    unsigned char RX_dummy;

    if(I2C_SR & I2C_SR_SRW) // Slave transmit?                               //D-decision
    {
        I2C_CR1 |= I2C_CR1_TX;//IICC_TX = 0b1; // I2C: TX mode
        I2C_DATA = TX1;   //IICD = TX1; // Load byte to transmit
        TX1 = 232 + State; // Load error code = "65000 + State"

        Pending_TX1 = 0;
    }
    else
    {
        I2C_CR1 &= ~I2C_CR1_TX;//IICC_TX = 0b0; // I2C: RX mode                                   //2-action
        RX_dummy = I2C_DATA; // Discard transmitted byte                     //3-action
    }
}




/* function definitions */

uint8_t IIC_set_bps(uint32_t bps)
{
  uint8_t x;
  uint8_t best_ndx=(uint8_t)-1u;
  uint16_t e=(uint16_t)-1u;
  uint32_t d=(I2C_CLK)/bps;
  
  for(x=0; x<sizeof(IIC_prescaler_val)/sizeof(IIC_prescaler_val[0]); x++)
  {
    uint16_t e1;
    if (d>IIC_prescaler_val[x])
    {
      continue;
    }
    
    e1=(uint16_t)(IIC_prescaler_val[x]-d);
    if (e1<e)
    {
      e=e1;
      best_ndx=x;
    }
  }

  if (best_ndx == (uint8_t)-1u)
  {
    return(1);
  }
  I2C_FREQDIV = best_ndx;
  return(0);
}

void IIC_set_FACK_flag(bool enable)
{
  if(enable)
  {
	  I2C_SMB_CSR |= I2C_SMB_CSR_FACK; /* enable FACK. */
	  FACK_Flag = 1;  	
  }
  else
  {
	  I2C_SMB_CSR &= ~I2C_SMB_CSR_FACK; /* disable FACK. */
	  FACK_Flag = 0;  	
  }
}

void IIC_set_RW_flag(bool read)
{
  if(read)
  {
 	RW_Flag  = IIC_RW_READ;
  }
  else
  {
 	RW_Flag  = IIC_RW_WRITE;
  }
}

/*****************************************************************************
* Funcion Name  : InitPorts
* Parameters    : none
* Returns       : none
* Notes         : 
******************************************************************************/

 void iic_init(void){
  
  IIC_init_drv(TData,10,RData,11);
  ENABLE_IIC();
  EnableIntrp();
  
  SET_7BIT_MODE(); /* set 7 bits addressing mode */

  SET_RECEIVE_MODE(); /* select slave mode */
  ENABLE_IIC_INTERRUPT();
      
  if(I2C_CR2 & I2C_CR2_ADEXT){ 
  	SET_SLAVE_10BIT_ADDR(SLAVE_10BIT_ADDR_1); 
  }    	
   
  else{ 
  	SET_SLAVE_7BIT_ADDR(SLAVE_7BIT_ADDR_1); /* set slave address */ 
  }      
  
  DISABLE_GENERAL_CALL(); /* not use general call */    
   
  ENABLE_IIC_INTERRUPT();
}


/*
 * IIC_init_drv: Initialize the internal driver structure and configure the GPIOB0/1
 *               pins as SCL/SDA, also enable the clock to I2C.
 *
 *
 * Parameters: 
 * txBuf  - pointer to the transmit buffer which contains data to be sent
 * txSize - the size of the data to be sent in bytes
 * rxBuf  - pointer to the receive buffer to store the received data
 * rxSize - the size of the data to receive in bytes
 *
 * Return : none
 */

void IIC_init_drv( int8_t* txBuf, int16_t txSize, int8_t* rxBuf, int16_t rxSize)
{
	base_txBuf = txp = txBuf;
	base_rxBuf = rxp = rxBuf;
	LastRdOffst = rxSize;
	LastWrOffst = txSize;
	
    SIM_GPSA    = SIM_GPSA|0x0080;
	SIM_GPSB0   = SIM_GPSB0|0x1800;
    
    GPIOA_PER |= GPIOA_PER_PER_6;
    GPIOB_PER |= GPIOB_PER_PER_4;
	
	/* Enable clock to I2C */
	SIM_PCE |= SIM_PCE_I2C;
	
}

bool IIC_is_read_done(void)
{
	return (rxp	>= base_rxBuf+LastRdOffst);
}

bool IIC_is_write_done(void)
{
	return (txp >= base_txBuf+LastWrOffst);
}


#pragma interrupt alignsp saveall 
void IIC_SMBusIsr (void)//FIIC_SMBusIsr
{
	IIC_SMBusIsrCallback();
}
 
#pragma interrupt alignsp called
void IIC_SMBusIsrCallback (void)
{
	if(!(I2C_SMB_CSR & I2C_SMB_CSR_SLTF))
	{
		IIC_IsrCallback();
	}
	else
	{
		CLEAR_IIC_INT();		
	}
}


#pragma interrupt alignsp called
void IIC_IsrCallback (void)
{
    unsigned char RX_dummy;

    CLEAR_IIC_INT();//IICS_IICIF = 0b1;  // I2C: Clear interrupt State                    //1-action

    if (!(I2C_CR1 & I2C_CR1_MST)) // Slave mode? if (IICC_MST == 0b0)                                 //A-decision
    {
        //------ Arbitration Y
        if (I2C_SR & I2C_SR_ARBL) // Arbitration lost?                      //B-decision
        {
            I2C_SR |= I2C_SR_ARBL;//IICS_ARBL = 0b1; // Clear ARBL

            if (I2C_SR & I2C_SR_IAAS) //if (IICS_IAAS == 0b1) Addressed as slave?                //C-decision
            {
                srw ();
               
            }
        }//------ Arbitration N
        else
        {
            //------- YAAS Y (Address)
            if (I2C_SR & I2C_SR_IAAS) // if (IICS_IAAS == 0b1) Addressed as slave? 
            {
                srw ();//---- Transmit/Receiv?
                rx_buf_read_pointer=0;
            }
            else//------- YAAS N (Data)
            {
                if (I2C_CR1 & I2C_CR1_TX) // TX? if (IICC_TX == 0b1)                                //E-decision
                {
                    if (I2C_SR & I2C_SR_RXAK) // Master sent ACK? if (IICS_RXAK == 0b0) 
                    {
                        I2C_DATA = TX2; // Load byte to transmit
                        TX2 = 253; // Load error code = "65000 + State"

                        Pending_TX2 = 0;
                        
                    }
                    else
                    {
                        I2C_CR1 &= ~I2C_CR1_TX; //IICC_TX = 0b0; // I2C: RX mode
                        RX_dummy = I2C_DATA; // Discard transmitted byte
                    }
                }
                else
                {
                    //RX = IICD; // Read transmitted byte
                    tempData = I2C_DATA;
                                                               //4-action
                    rxbuffer[rx_buf_read_pointer++] = tempData;
                    Data_Received = 1;
                    //Print(tempData);
                    
                    if(tempData=='#'){
                    	
                    	rx_buf_read_pointer=0;
                    	Print(rxbuffer);
                    	
                    }
                      
                }
            }
        }
    }

}

#endif
